<?php
/**
 * Plugin Name: LHT Google Ads API Pro
 * Plugin URI: https://lehongtuan.com
 * Description: Server-side Tracking: Đẩy chuyển đổi Purchase về Google Ads. Fix lỗi trùng lặp đơn hàng (Deduplication).
 * Version: 1.1.2
 * Author: Lê Hồng Tuân
 * License: GPLv2
 */

if ( ! defined( 'ABSPATH' ) ) exit;

/**
 * -------------------------------------------------------------------
 * AUTO UPDATE
 * -------------------------------------------------------------------
 */
if ( ! class_exists( '\YahnisElsts\PluginUpdateChecker\v5\PucFactory' ) ) {
    require plugin_dir_path( __FILE__ ) . 'plugin-update-checker/plugin-update-checker.php';
}

use YahnisElsts\PluginUpdateChecker\v5\PucFactory;

$lht_gads_update_checker = PucFactory::buildUpdateChecker(
    'https://plugin-update.lehongtuan.com/lht-gads-api/lht-gads-api.json',
    __FILE__,
    'lht-gads-api'
);

/**
 * -------------------------------------------------------------------
 * LOGIC GOOGLE ADS API
 * -------------------------------------------------------------------
 */

if ( ! class_exists( 'LHT_GAds_API' ) ) :

class LHT_GAds_API {

    private $option_name = 'lht_gads_api_settings';
    private $cookie_name = 'gclid';
    private $meta_key    = '_google_click_id';
    private $log_file;
    private $api_version = 'v19'; 

    public function __construct() {
        $upload_dir = wp_upload_dir();
        $this->log_file = $upload_dir['basedir'] . '/lht_gads_events.log';

        add_action( 'admin_menu', array( $this, 'add_admin_menu' ) );
        add_action( 'admin_init', array( $this, 'register_settings' ) );
        add_action( 'admin_init', array( $this, 'handle_clear_log' ) );
        add_action( 'admin_init', array( $this, 'handle_oauth_callback' ) );

        add_action( 'init', array( $this, 'set_gclid_cookie' ) );
        add_action( 'woocommerce_checkout_update_order_meta', array( $this, 'save_gclid_to_order' ) );
        add_action( 'woocommerce_order_status_completed', array( $this, 'send_conversion_event' ), 10, 1 );
    }

    public function add_admin_menu() {
        add_options_page('LHT Google Ads API', 'LHT GAds API', 'manage_options', 'lht-gads-api', array( $this, 'settings_page' ));
    }

    public function register_settings() {
        register_setting( 'lht_gads_api_group', $this->option_name );
    }

    private function write_log( $message ) {
        $opt = get_option( $this->option_name );
        if ( empty( $opt['enable_log'] ) ) return;
        $time = current_time( 'mysql' );
        file_put_contents( $this->log_file, "[{$time}] {$message}" . PHP_EOL, FILE_APPEND );
    }

    private function send_error_email( $order_id, $error_msg ) {
        $opt = get_option( $this->option_name );
        if ( empty( $opt['enable_error_email'] ) ) return;

        $admin_email = get_option( 'admin_email' );
        $site_name = get_bloginfo( 'name' );
        $subject = "[LHT GAds Alert] ⚠️ Lỗi đẩy API đơn hàng #{$order_id}";
        
        $body  = "Xin chào Admin,\n\n";
        $body .= "Hệ thống Google Ads API Tracking trên website {$site_name} gặp lỗi.\n";
        $body .= "--------------------------------------------------\n";
        $body .= "🆔 Đơn hàng: #{$order_id}\n";
        $body .= "⏰ Thời gian: " . current_time('mysql') . "\n";
        $body .= "❌ Chi tiết lỗi: \n{$error_msg}\n";
        $body .= "--------------------------------------------------\n";

        wp_mail( $admin_email, $subject, $body );
        $this->write_log( "📧 Đã gửi email cảnh báo lỗi tới: {$admin_email}" );
    }

    public function handle_clear_log() {
        if ( isset( $_POST['lht_clear_log_btn'] ) && current_user_can( 'manage_options' ) ) {
            file_put_contents( $this->log_file, "" );
            add_settings_error( 'lht_gads_api_group', 'log_cleared', 'Đã xóa toàn bộ log.', 'success' );
        }
    }

    private function get_redirect_uri() {
        return admin_url( 'options-general.php?page=lht-gads-api' );
    }

    public function handle_oauth_callback() {
        if ( isset( $_GET['page'] ) && $_GET['page'] == 'lht-gads-api' && isset( $_GET['code'] ) ) {
            if ( ! current_user_can( 'manage_options' ) ) return;

            $opt = get_option( $this->option_name );
            $code = sanitize_text_field( $_GET['code'] );
            $redirect_uri = $this->get_redirect_uri();

            $url = 'https://oauth2.googleapis.com/token';
            $body = array(
                'client_id'     => $opt['client_id'],
                'client_secret' => $opt['client_secret'],
                'code'          => $code,
                'grant_type'    => 'authorization_code',
                'redirect_uri'  => $redirect_uri,
            );

            $response = wp_remote_post( $url, array( 'body' => $body ) );

            if ( is_wp_error( $response ) ) {
                add_settings_error( 'lht_gads_api_group', 'oauth_error', 'Lỗi kết nối: ' . $response->get_error_message(), 'error' );
            } else {
                $data = json_decode( wp_remote_retrieve_body( $response ), true );
                if ( isset( $data['error'] ) ) {
                    add_settings_error( 'lht_gads_api_group', 'oauth_api_error', 'Google Lỗi: ' . $data['error_description'], 'error' );
                } elseif ( isset( $data['refresh_token'] ) ) {
                    $opt['refresh_token'] = $data['refresh_token'];
                    update_option( $this->option_name, $opt );
                    add_settings_error( 'lht_gads_api_group', 'oauth_success', '✅ Kết nối thành công! Đã lấy được Refresh Token.', 'success' );
                    wp_redirect( remove_query_arg( array('code', 'scope', 'state'), $redirect_uri ) );
                    exit;
                } else {
                     add_settings_error( 'lht_gads_api_group', 'oauth_no_refresh', 'Thành công nhưng không có Refresh Token. Vui lòng thử lại và nhớ bấm "Cho phép" tất cả.', 'warning' );
                }
            }
        }
    }

    public function settings_page() {
        $opt = get_option( $this->option_name );
        $log_content = file_exists( $this->log_file ) ? file_get_contents( $this->log_file ) : 'Chưa có dữ liệu...';
        
        $redirect_uri = $this->get_redirect_uri();
        $has_refresh_token = !empty($opt['refresh_token']);
        
        $auth_url = '#';
        if (!empty($opt['client_id'])) {
            $params = array(
                'client_id'     => $opt['client_id'],
                'redirect_uri'  => $redirect_uri,
                'response_type' => 'code',
                'scope'         => 'https://www.googleapis.com/auth/adwords',
                'access_type'   => 'offline',
                'prompt'        => 'consent',
            );
            $auth_url = 'https://accounts.google.com/o/oauth2/v2/auth?' . http_build_query($params);
        }

        $timezones = array(
            '+07:00' => '(GMT+07:00) Vietnam, Bangkok, Jakarta',
            '+00:00' => '(GMT+00:00) UTC',
            '-05:00' => '(GMT-05:00) Eastern Time (US & Canada)',
            '-08:00' => '(GMT-08:00) Pacific Time (US & Canada)',
            '+08:00' => '(GMT+08:00) Beijing, Singapore',
            '+09:00' => '(GMT+09:00) Tokyo, Seoul',
        );
        $current_tz = $opt['timezone'] ?? '+07:00';
        ?>
        <div class="wrap">
            <h1>🚀 LHT Google Ads API (Version 1.1.2)</h1>
            <?php settings_errors(); ?>

            <div style="background: #fff; border: 1px solid #c3c4c7; padding: 0; margin-bottom: 20px; max-width: 1200px; box-shadow: 0 1px 1px rgba(0,0,0,.04);">
                <h2 style="margin: 0; padding: 15px 20px; background: #f0f0f1; border-bottom: 1px solid #c3c4c7; font-size: 14px;">📖 Hướng dẫn cấu hình</h2>
                <div style="padding: 20px;">
                    <ol style="margin-top:0; margin-left: 20px; line-height: 1.6;">
                        <li><strong>Bước 1:</strong> Truy cập <a href="https://console.cloud.google.com/" target="_blank">Google Cloud Console</a>.
                            <ul style="list-style: circle; margin-left: 20px;">
                                <li>Tạo <strong>Project mới</strong> > Enable <strong>Google Ads API</strong>.</li>
                            </ul>
                        </li>
                        <li><strong>Bước 2:</strong> Vào <strong>OAuth consent screen</strong>.
                            <ul style="list-style: circle; margin-left: 20px;">
                                <li>User Type: <strong>External</strong>. Sau khi tạo xong nhớ bấm <strong>PUBLISH APP</strong>.</li>
                            </ul>
                        </li>
                        <li><strong>Bước 3:</strong> Vào <strong>Credentials</strong> > Create <strong>OAuth client ID</strong>.
                            <ul style="list-style: circle; margin-left: 20px;">
                                <li>Type: <strong>Web application</strong>.</li>
                                <li>Authorized redirect URIs (BẮT BUỘC): <code style="background: #eef; padding: 3px 5px;"><?php echo $redirect_uri; ?></code></li>
                            </ul>
                        </li>
                        <li><strong>Bước 4:</strong> Copy Client ID & Secret dán vào dưới > Bấm <strong>Lưu Cấu Hình</strong> > Rồi bấm <strong>Kết nối Google</strong>.</li>
                    </ol>
                </div>
            </div>
            
            <div style="display: flex; gap: 20px; flex-wrap: wrap;">
                <div style="flex: 1; min-width: 400px;">
                    <form method="post" action="options.php" style="background: #fff; padding: 20px; border: 1px solid #c3c4c7;">
                        <?php settings_fields( 'lht_gads_api_group' ); ?>
                        <h2 style="margin-top:0;">📡 Cấu hình thông số</h2>
                        <table class="form-table">
                            <tr>
                                <th scope="row">Developer Token</th>
                                <td><input type="text" name="<?php echo $this->option_name; ?>[dev_token]" value="<?php echo esc_attr($opt['dev_token'] ?? ''); ?>" class="large-text code" /></td>
                            </tr>
                            <tr>
                                <th scope="row">Client ID (App)</th>
                                <td><input type="text" name="<?php echo $this->option_name; ?>[client_id]" value="<?php echo esc_attr($opt['client_id'] ?? ''); ?>" class="large-text code" /></td>
                            </tr>
                            <tr>
                                <th scope="row">Client Secret</th>
                                <td><input type="text" name="<?php echo $this->option_name; ?>[client_secret]" value="<?php echo esc_attr($opt['client_secret'] ?? ''); ?>" class="large-text code" /></td>
                            </tr>
                            <tr>
                                <th scope="row">Kết nối Google</th>
                                <td>
                                    <?php if ($has_refresh_token): ?>
                                        <div style="color: green; margin-bottom: 5px;"><strong>✅ Đã kết nối thành công!</strong></div>
                                    <?php endif; ?>
                                    
                                    <?php if (!empty($opt['client_id'])): ?>
                                        <a href="<?php echo $auth_url; ?>" class="button button-primary" style="background: #db4437; border-color: #db4437;">
                                            <?php echo $has_refresh_token ? '🔄 Kết nối lại với Google' : 'G+ Kết nối với Google'; ?>
                                        </a>
                                    <?php else: ?>
                                        <p class="description" style="color: red;">Vui lòng nhập Client ID & Secret rồi Lưu lại để hiện nút kết nối.</p>
                                    <?php endif; ?>

                                    <input type="hidden" name="<?php echo $this->option_name; ?>[refresh_token]" value="<?php echo esc_attr($opt['refresh_token'] ?? ''); ?>" />
                                </td>
                            </tr>
                            <tr>
                                <th scope="row">Customer ID (Ads)</th>
                                <td>
                                    <input type="text" name="<?php echo $this->option_name; ?>[customer_id]" value="<?php echo esc_attr($opt['customer_id'] ?? ''); ?>" class="regular-text" placeholder="ID tài khoản con (VD: 123-456-7890)" />
                                </td>
                            </tr>
                            <tr style="background: #f0f6fc;">
                                <th scope="row">MCC ID (Login Customer)</th>
                                <td>
                                    <input type="text" name="<?php echo $this->option_name; ?>[login_customer_id]" value="<?php echo esc_attr($opt['login_customer_id'] ?? ''); ?>" class="regular-text" placeholder="ID tài khoản quản lý (Nếu có)" />
                                </td>
                            </tr>
                            <tr>
                                <th scope="row">Conversion Action ID</th>
                                <td><input type="text" name="<?php echo $this->option_name; ?>[conv_id]" value="<?php echo esc_attr($opt['conv_id'] ?? ''); ?>" class="regular-text" /></td>
                            </tr>
                            <tr>
                                <th scope="row">Múi giờ tài khoản</th>
                                <td>
                                    <select name="<?php echo $this->option_name; ?>[timezone]" style="width: 100%; max-width: 400px;">
                                        <?php foreach ($timezones as $val => $label): ?>
                                            <option value="<?php echo esc_attr($val); ?>" <?php selected($current_tz, $val); ?>><?php echo esc_html($label); ?></option>
                                        <?php endforeach; ?>
                                    </select>
                                </td>
                            </tr>
                            <tr>
                                <th scope="row" style="color: #d63638;">🔥 Chuyển đổi nâng cao</th>
                                <td>
                                    <label>
                                        <input type="checkbox" name="<?php echo $this->option_name; ?>[enable_enhanced]" value="1" <?php checked( 1, $opt['enable_enhanced'] ?? 0 ); ?> />
                                        Gửi kèm <strong>Email & SĐT</strong> (Hashed SHA256)
                                    </label>
                                </td>
                            </tr>
                            <tr>
                                <th scope="row">Cảnh báo & Debug</th>
                                <td>
                                    <label style="display:block; margin-bottom: 8px;">
                                        <input type="checkbox" name="<?php echo $this->option_name; ?>[enable_error_email]" value="1" <?php checked( 1, $opt['enable_error_email'] ?? 0 ); ?> />
                                        <strong>Gửi Email cho Admin khi gặp lỗi API</strong>
                                    </label>
                                    <label>
                                        <input type="checkbox" name="<?php echo $this->option_name; ?>[enable_log]" value="1" <?php checked( 1, $opt['enable_log'] ?? 0 ); ?> /> 
                                        Ghi nhật ký hoạt động (Debug)
                                    </label>
                                </td>
                            </tr>
                        </table>
                        <hr>
                        <?php submit_button('Lưu Cấu Hình'); ?>
                    </form>
                </div>

                <div style="flex: 1; min-width: 400px;">
                    <div style="background: #fff; padding: 20px; border: 1px solid #c3c4c7;">
                        <h2 style="margin-top:0; display:flex; justify-content:space-between;">
                            📝 Live Logs
                            <form method="post" style="display:inline;"><input type="submit" name="lht_clear_log_btn" class="button" value="Xóa Log" /></form>
                        </h2>
                        <textarea readonly style="width: 100%; height: 500px; background: #0c0d0e; color: #00ff00; font-family: monospace; padding: 10px;"><?php echo esc_textarea( $log_content ); ?></textarea>
                    </div>
                </div>
            </div>
        </div>
        <?php
    }

    public function set_gclid_cookie() {
        if ( isset( $_GET['gclid'] ) ) {
            setcookie( $this->cookie_name, sanitize_text_field( $_GET['gclid'] ), time() + ( 90 * 86400 ), COOKIEPATH, COOKIE_DOMAIN );
        }
    }

    public function save_gclid_to_order( $order_id ) {
        if ( isset( $_COOKIE[$this->cookie_name] ) ) {
            update_post_meta( $order_id, $this->meta_key, sanitize_text_field( $_COOKIE[$this->cookie_name] ) );
        }
    }

    private function normalize_and_hash( $value, $type ) {
        if ( empty($value) ) return null;
        if ( $type === 'email' ) {
            $normalized = trim( strtolower( $value ) );
        } elseif ( $type === 'phone' ) {
            $digits = preg_replace( '/[^0-9]/', '', $value );
            if ( substr( $digits, 0, 1 ) === '0' ) {
                $digits = '84' . substr( $digits, 1 );
            }
            $normalized = '+' . $digits; 
        } else {
            return null;
        }
        return hash( 'sha256', $normalized );
    }

    private function get_formatted_conversion_time( $timezone_offset ) {
        $timestamp_utc = time(); 
        
        preg_match('/([+-])(\d{2}):(\d{2})/', $timezone_offset, $matches);
        if (!empty($matches)) {
            $sign = $matches[1];
            $hours = (int)$matches[2];
            $mins = (int)$matches[3];
            $offset_seconds = ($hours * 3600) + ($mins * 60);
            if ($sign == '-') $offset_seconds *= -1;
            $timestamp_local = $timestamp_utc + $offset_seconds;
        } else {
            $timestamp_local = $timestamp_utc; 
        }

        return gmdate('Y-m-d H:i:s', $timestamp_local) . $timezone_offset;
    }

    public function send_conversion_event( $order_id ) {
        $opt = get_option( $this->option_name );
        
        if ( empty($opt['dev_token']) ) {
            $this->write_log( "❌ Chưa cấu hình API." );
            return;
        }

        $gclid = get_post_meta( $order_id, $this->meta_key, true );
        if ( ! $gclid ) {
            $this->write_log( "⚠️ Bỏ qua đơn #{$order_id}: Không có GCLID." );
            return;
        }

        $order = wc_get_order( $order_id );
        $this->write_log( "🔄 Đang xử lý đơn #{$order_id} (Total: " . $order->get_total() . ")" );

        $access_token = $this->get_access_token( $opt );
        if ( ! $access_token ) {
            $msg = "Không lấy được Access Token (Vui lòng bấm Kết nối lại với Google).";
            $this->write_log( "❌ Lỗi: " . $msg );
            $this->send_error_email( $order_id, $msg );
            return;
        }

        $customer_id_clean = str_replace( '-', '', $opt['customer_id'] );
        $conversion_action_id = trim($opt['conv_id']);
        
        $headers = array(
            'Authorization' => 'Bearer ' . $access_token,
            'developer-token' => $opt['dev_token'],
            'Content-Type' => 'application/json',
        );

        if ( ! empty( $opt['login_customer_id'] ) ) {
            $login_id_clean = str_replace( '-', '', $opt['login_customer_id'] );
            $headers['login-customer-id'] = $login_id_clean;
        }

        $resource_name = "customers/{$customer_id_clean}/conversionActions/{$conversion_action_id}";
        
        $timezone_offset = $opt['timezone'] ?? '+07:00';
        $conversion_time = $this->get_formatted_conversion_time($timezone_offset);

        // --- Cập nhật: Thêm orderId để chống trùng lặp ---
        $conversion_data = array(
            'gclid' => $gclid,
            'conversionAction' => $resource_name,
            'conversionDateTime' => $conversion_time,
            'conversionValue' => (float) $order->get_total(),
            'currencyCode' => $order->get_currency(),
            'orderId' => (string) $order_id, // Fix deduplication
        );

        if ( ! empty( $opt['enable_enhanced'] ) ) {
            $hashed_email = $this->normalize_and_hash( $order->get_billing_email(), 'email' );
            $hashed_phone = $this->normalize_and_hash( $order->get_billing_phone(), 'phone' );
            
            $identifiers = array();
            if ( $hashed_email ) {
                $identifiers[] = array( 'userIdentifierSource' => 'FIRST_PARTY', 'hashedEmail' => $hashed_email );
            }
            if ( $hashed_phone ) {
                $identifiers[] = array( 'userIdentifierSource' => 'FIRST_PARTY', 'hashedPhoneNumber' => $hashed_phone );
            }

            if ( ! empty( $identifiers ) ) {
                $conversion_data['userIdentifiers'] = $identifiers;
                $this->write_log( "🔹 Đã đính kèm dữ liệu nâng cao." );
            }
        }

        $payload = array(
            'conversions' => array( $conversion_data ),
            'partialFailure' => true,
        );

        $url = "https://googleads.googleapis.com/{$this->api_version}/customers/{$customer_id_clean}:uploadClickConversions";

        $response = wp_remote_post( $url, array(
            'headers' => $headers,
            'body' => json_encode( $payload ),
            'timeout' => 45,
        ));

        if ( is_wp_error( $response ) ) {
            $error_msg = $response->get_error_message();
            $this->write_log( '❌ CURL Error: ' . $error_msg );
            $this->send_error_email( $order_id, "CURL Error: " . $error_msg );
        } else {
            $body_string = wp_remote_retrieve_body($response);
            $body = json_decode($body_string, true);
            
            if(isset($body['error'])) {
                 $msg = "Code: " . $body['error']['code'] . " - " . $body['error']['message'];
                 $this->write_log( "❌ API Lỗi: {$msg}" );
                 $this->send_error_email( $order_id, $msg );

            } elseif(isset($body['partialFailureError'])) {
                 $error_details = print_r($body['partialFailureError'], true);
                 // Check if error is "Order ID already exists"
                 if (strpos($error_details, 'CLICK_CONVERSION_ALREADY_EXISTS') !== false) {
                     $this->write_log( "⚠️ Đơn hàng #{$order_id} đã tồn tại trên Google Ads (Đã chặn trùng lặp)." );
                 } else {
                     $this->write_log( "❌ API Lỗi Partial: " . $error_details );
                     $this->send_error_email( $order_id, "Partial Failure: " . $error_details );
                 }

            } elseif (isset($body['results'])) {
                 $this->write_log( "✅ THÀNH CÔNG: Đã đẩy đơn #{$order_id} (Time: {$conversion_time})." );

            } else {
                 $msg = substr($body_string, 0, 200);
                 $this->write_log( "⚠️ Phản hồi lạ: " . $msg );
                 $this->send_error_email( $order_id, "Unknown Response: " . $msg );
            }
        }
    }

    private function get_access_token( $opt ) {
        if ( empty($opt['refresh_token']) ) return false;
        
        $url = 'https://oauth2.googleapis.com/token';
        $body = array(
            'client_id' => $opt['client_id'],
            'client_secret' => $opt['client_secret'],
            'refresh_token' => $opt['refresh_token'],
            'grant_type' => 'refresh_token',
        );
        $response = wp_remote_post( $url, array( 'body' => $body ) );
        if ( is_wp_error( $response ) ) return false;
        $body = json_decode( wp_remote_retrieve_body( $response ), true );
        return $body['access_token'] ?? false;
    }
}

new LHT_GAds_API();

endif;