<?php
/**
 * Plugin Name: LHT Meta Dataset Tracker
 * Plugin URI: https://lehongtuan.com
 * Description: Gửi sự kiện Purchase từ WooCommerce sang Meta Dataset (Conversion API) chỉ khi đơn hàng thanh toán thành công. Lưu fbp/fbc, hash email/phone, cấu hình trong admin.
 * Author: Lê Hồng Tuân
 * Author URI: https://lehongtuan.com
 * Version: 1.0.2
 */

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Tự động cập nhật plugin (self-hosted) với Plugin Update Checker
 * - Yêu cầu thư mục: plugin-update-checker/ nằm cùng cấp file plugin này
 * - File JSON update: https://plugin-update.lehongtuan.com/lht-meta-dataset-tracker/lht-meta-dataset-tracker.json
 */
if (!class_exists('\YahnisElsts\PluginUpdateChecker\v5\PucFactory')) {
    require plugin_dir_path(__FILE__) . 'plugin-update-checker/plugin-update-checker.php';
}

use YahnisElsts\PluginUpdateChecker\v5\PucFactory;

$ht_meta_update_checker = PucFactory::buildUpdateChecker(
    'https://plugin-update.lehongtuan.com/lht-meta-dataset-tracker/lht-meta-dataset-tracker.json', // URL JSON metadata
    __FILE__,                                                                                       // File chính plugin
    'lht-meta-dataset-tracker'                                                                      // Slug duy nhất
);

if (!class_exists('HT_Meta_Dataset_Tracker')):

    class HT_Meta_Dataset_Tracker
    {

        const OPTION_KEY = 'ht_meta_dataset_settings';

        public function __construct()
        {
            // Trang cài đặt
            add_action('admin_menu', [$this, 'add_settings_page']);
            add_action('admin_init', [$this, 'register_settings']);

            // Lưu fbp/fbc vào order meta
            add_action('woocommerce_checkout_update_order_meta', [$this, 'capture_fb_cookies_to_order_meta']);

            // Gửi event khi đổi trạng thái đơn
            add_action('woocommerce_order_status_changed', [$this, 'maybe_send_purchase_event'], 10, 4);

            // CSS admin cho trang setting
            add_action('admin_head', [$this, 'admin_settings_css']);
        }

        /**
         * Thiết lập mặc định khi kích hoạt plugin
         */
        public static function activate()
        {
            $defaults = [
                'enabled' => 0,
                'dataset_id' => '',
                'access_token' => '',
                'order_status' => 'completed', // completed | processing
                'enable_logging' => 0,
                'test_event_code' => '',
            ];

            $current = get_option(self::OPTION_KEY, []);
            $merged = wp_parse_args($current, $defaults);

            update_option(self::OPTION_KEY, $merged);
        }

        /**
         * Thêm trang cài đặt
         */
        public function add_settings_page()
        {
            add_options_page(
                'LHT Meta Dataset',
                'LHT Meta Dataset',
                'manage_options',
                'ht-meta-dataset',
                [$this, 'render_settings_page']
            );
        }

        /**
         * Đăng ký settings
         */
        public function register_settings()
        {
            register_setting(
                'ht_meta_dataset_group',
                self::OPTION_KEY,
                [$this, 'sanitize_settings']
            );

            add_settings_section(
                'ht_meta_dataset_main',
                'Cấu hình Meta Dataset (Conversion API)',
                function () {
                    echo '<p>Nhập Dataset ID &amp; Access Token từ Meta Events Manager. Plugin chỉ gửi sự kiện <strong>Purchase</strong> khi đơn hàng đạt trạng thái được chọn.</p>';
                },
                'ht-meta-dataset'
            );

            // Bật / tắt
            add_settings_field(
                'enabled',
                'Bật gửi sự kiện',
                function () {
                    $options = $this->get_options();
                    ?>
                <label>
                    <input type="checkbox" name="<?php echo esc_attr(self::OPTION_KEY); ?>[enabled]" value="1" <?php checked(1, $options['enabled'] ?? 0); ?> />
                    Kích hoạt gửi Purchase lên Meta Dataset
                </label>
                <?php
                },
                'ht-meta-dataset',
                'ht_meta_dataset_main'
            );

            // Dataset ID
            add_settings_field(
                'dataset_id',
                'Dataset ID',
                function () {
                    $options = $this->get_options();
                    ?>
                <input type="text" name="<?php echo esc_attr(self::OPTION_KEY); ?>[dataset_id]"
                    value="<?php echo esc_attr($options['dataset_id'] ?? ''); ?>" class="regular-text"
                    placeholder="Ví dụ: 123456789012345" />
                <p class="description">
                    Dataset ID lấy trong Events Manager &rarr; Datasets.
                </p>
                <?php
                },
                'ht-meta-dataset',
                'ht_meta_dataset_main'
            );

            // Access Token
            add_settings_field(
                'access_token',
                'Access Token',
                function () {
                    $options = $this->get_options();
                    ?>
                <input type="password" name="<?php echo esc_attr(self::OPTION_KEY); ?>[access_token]"
                    value="<?php echo esc_attr($options['access_token'] ?? ''); ?>" class="regular-text"
                    placeholder="Access Token server-side" />
                <p class="description">
                    Dùng token server-side riêng, tạo trong Events Manager &rarr; Settings &rarr; Conversions API.
                </p>
                <?php
                },
                'ht-meta-dataset',
                'ht_meta_dataset_main'
            );

            // Test Event Code
            add_settings_field(
                'test_event_code',
                'Test Event Code (tùy chọn)',
                function () {
                    $options = $this->get_options();
                    ?>
                <input type="text" name="<?php echo esc_attr(self::OPTION_KEY); ?>[test_event_code]"
                    value="<?php echo esc_attr($options['test_event_code'] ?? ''); ?>" class="regular-text"
                    placeholder="Nhập mã Test Event từ tab Test Events nếu muốn xem realtime" />
                <p class="description">
                    Dùng để xem log realtime trong Test Events. Khi chạy thật nên để <strong>trống</strong> để tránh double-events.
                </p>
                <?php
                },
                'ht-meta-dataset',
                'ht_meta_dataset_main'
            );

            // Order Status
            add_settings_field(
                'order_status',
                'Trạng thái đơn để gửi Purchase',
                function () {
                    $options = $this->get_options();
                    $order_status = $options['order_status'] ?? 'completed';
                    ?>
                <select name="<?php echo esc_attr(self::OPTION_KEY); ?>[order_status]">
                    <option value="completed" <?php selected($order_status, 'completed'); ?>>
                        Chỉ khi đơn "Completed"
                    </option>
                    <option value="processing" <?php selected($order_status, 'processing'); ?>>
                        Khi đơn "Processing"
                    </option>
                </select>
                <p class="description">
                    <strong>Gợi ý:</strong> Nếu có COD, nên chọn <code>Completed</code>. Nếu chỉ thanh toán online, có thể chọn
                    <code>Processing</code>.
                </p>
                <?php
                },
                'ht-meta-dataset',
                'ht_meta_dataset_main'
            );

            // Logging
            add_settings_field(
                'enable_logging',
                'Bật ghi log (debug)',
                function () {
                    $options = $this->get_options();
                    ?>
                <label>
                    <input type="checkbox" name="<?php echo esc_attr(self::OPTION_KEY); ?>[enable_logging]" value="1" <?php checked(1, $options['enable_logging'] ?? 0); ?> />
                    Ghi log vào <code>wp-content/debug.log</code> (cần bật WP_DEBUG &amp; WP_DEBUG_LOG).
                </label>
                <p class="description">
                    Chỉ nên bật khi cần debug, tránh để luôn ở môi trường production.
                </p>
                <?php
                },
                'ht-meta-dataset',
                'ht_meta_dataset_main'
            );
        }

        /**
         * Sanitization
         */
        public function sanitize_settings($input)
        {
            $output = [];

            $output['enabled'] = !empty($input['enabled']) ? 1 : 0;
            $output['dataset_id'] = isset($input['dataset_id']) ? sanitize_text_field($input['dataset_id']) : '';
            $output['access_token'] = isset($input['access_token']) ? trim($input['access_token']) : '';
            $output['order_status'] = in_array($input['order_status'] ?? 'completed', ['completed', 'processing'], true)
                ? $input['order_status']
                : 'completed';
            $output['enable_logging'] = !empty($input['enable_logging']) ? 1 : 0;
            $output['test_event_code'] = isset($input['test_event_code']) ? sanitize_text_field($input['test_event_code']) : '';

            return $output;
        }

        /**
         * Lấy options với default
         */
        private function get_options()
        {
            $defaults = [
                'enabled' => 0,
                'dataset_id' => '',
                'access_token' => '',
                'order_status' => 'completed',
                'enable_logging' => 0,
                'test_event_code' => '',
            ];

            $options = get_option(self::OPTION_KEY, []);

            return wp_parse_args($options, $defaults);
        }

        /**
         * CSS cho trang settings (card layout)
         */
        public function admin_settings_css()
        {
            $screen = function_exists('get_current_screen') ? get_current_screen() : null;
            if (!$screen || $screen->id !== 'settings_page_ht-meta-dataset') {
                return;
            }
            ?>
            <style>
                .ht-meta-wrap {
                    max-width: 900px;
                }

                .ht-meta-header {
                    margin-bottom: 20px;
                }

                .ht-meta-title {
                    font-size: 24px;
                    font-weight: 600;
                    margin-bottom: 4px;
                }

                .ht-meta-badge {
                    display: inline-block;
                    padding: 2px 8px;
                    border-radius: 999px;
                    background: #f3f4f6;
                    font-size: 11px;
                    font-weight: 500;
                    color: #555d66;
                    margin-left: 8px;
                }

                .ht-meta-subtitle {
                    color: #555d66;
                    font-size: 13px;
                    max-width: 650px;
                }

                .ht-meta-card {
                    background: #fff;
                    border-radius: 8px;
                    border: 1px solid #e5e5e5;
                    box-shadow: 0 1px 3px rgba(0, 0, 0, 0.04);
                    padding: 24px 24px 16px;
                    margin-bottom: 16px;
                }

                .ht-meta-card h2 {
                    margin-top: 0;
                    font-size: 16px;
                    margin-bottom: 16px;
                }

                .ht-meta-wrap .form-table {
                    margin-top: 0;
                }

                .ht-meta-wrap .form-table th {
                    width: 220px;
                    padding: 10px 10px 10px 0;
                }

                .ht-meta-wrap .form-table td {
                    padding: 10px 0;
                }

                .ht-meta-wrap input[type="text"],
                .ht-meta-wrap input[type="password"],
                .ht-meta-wrap select {
                    max-width: 100%;
                }

                .ht-meta-footer-note {
                    font-size: 12px;
                    color: #6c757d;
                    margin-top: 10px;
                }
            </style>
            <?php
        }

        /**
         * Giao diện trang settings
         */
        public function render_settings_page()
        {
            if (!current_user_can('manage_options')) {
                return;
            }
            ?>
            <div class="wrap ht-meta-wrap">
                <div class="ht-meta-header">
                    <div class="ht-meta-title">
                        LHT Meta Dataset Tracker
                        <span class="ht-meta-badge">v1.0.1</span>
                    </div>
                    <p class="ht-meta-subtitle">
                        Gửi sự kiện <strong>Purchase</strong> từ WooCommerce sang Meta Dataset (Conversions API),
                        chỉ khi đơn hàng đã thanh toán thành công. Tự động hash email/phone theo chuẩn, gửi kèm fbp/fbc nếu có.
                    </p>
                </div>

                <form method="post" action="options.php">
                    <?php
                    settings_fields('ht_meta_dataset_group');
                    do_settings_sections('ht-meta-dataset');
                    ?>
                    <div class="ht-meta-card">
                        <p class="ht-meta-footer-note">
                            Kiểm tra lại trong Events Manager &rarr; Test Events sau khi cấu hình xong.
                            Nếu có lỗi, bật ghi log để xem chi tiết trong <code>wp-content/debug.log</code>.
                        </p>
                    </div>
                    <?php submit_button('Lưu cấu hình Meta Dataset'); ?>
                </form>
            </div>
            <?php
        }

        /**
         * Lưu _fbp và _fbc vào order meta khi checkout
         */
        public function capture_fb_cookies_to_order_meta($order_id)
        {
            if (isset($_COOKIE['_fbp'])) {
                update_post_meta(
                    $order_id,
                    '_fbp',
                    sanitize_text_field(wp_unslash($_COOKIE['_fbp']))
                );
            }
            if (isset($_COOKIE['_fbc'])) {
                update_post_meta(
                    $order_id,
                    '_fbc',
                    sanitize_text_field(wp_unslash($_COOKIE['_fbc']))
                );
            }
        }

        /**
         * Xử lý khi đổi trạng thái đơn
         */
        public function maybe_send_purchase_event($order_id, $old_status, $new_status, $order)
        {
            $options = $this->get_options();

            if (empty($options['enabled'])) {
                return;
            }

            $target_status = $options['order_status']; // completed | processing

            if ($target_status === 'completed' && $new_status !== 'completed') {
                return;
            }

            if ($target_status === 'processing' && $new_status !== 'processing') {
                return;
            }

            $this->send_purchase_event_to_meta($order_id, $order, $options);
        }

        /**
         * Gửi Purchase tới Meta Dataset / CAPI
         */
        private function send_purchase_event_to_meta($order_id, $order, $options)
        {
            $dataset_id = $options['dataset_id'];
            $access_token = $options['access_token'];

            if (empty($dataset_id) || empty($access_token)) {
                $this->log('Dataset ID hoặc Access Token chưa cấu hình.', $options);
                return;
            }

            if (!class_exists('WC_Order')) {
                $this->log('WooCommerce chưa kích hoạt.', $options);
                return;
            }

            if (!$order instanceof WC_Order) {
                $order = wc_get_order($order_id);
                if (!$order) {
                    $this->log('Không tìm thấy order ' . $order_id, $options);
                    return;
                }
            }

            // fbp / fbc
            $fbp = get_post_meta($order_id, '_fbp', true);
            $fbc = get_post_meta($order_id, '_fbc', true);

            // User data
            $email = $order->get_billing_email();
            $phone = $order->get_billing_phone();
            $phone = preg_replace('/\D+/', '', (string) $phone);

            $user_data = [];

            if (!empty($email)) {
                $user_data['em'] = hash('sha256', strtolower(trim($email)));
            }
            if (!empty($phone)) {
                $user_data['ph'] = hash('sha256', $phone);
            }
            if (!empty($fbp)) {
                $user_data['fbp'] = $fbp;
            }
            if (!empty($fbc)) {
                $user_data['fbc'] = $fbc;
            }

            if (empty($user_data)) {
                $this->log('Cảnh báo: Không có user_data cho order ' . $order_id, $options);
            }

            // Contents
            $contents = [];
            foreach ($order->get_items() as $item) {
                if (!is_a($item, 'WC_Order_Item_Product')) {
                    continue;
                }
                $product_id = $item->get_product_id();
                if (!$product_id) {
                    continue;
                }
                $contents[] = [
                    'id' => (string) $product_id,
                    'quantity' => (int) $item->get_quantity(),
                ];
            }

            $event_time = time();
            if ($order->get_date_completed()) {
                $event_time = $order->get_date_completed()->getTimestamp();
            }

            $event = [
                'event_name' => 'Purchase',
                'event_time' => $event_time,
                'action_source' => 'website',
                'event_source_url' => $order->get_checkout_order_received_url(),
                'custom_data' => [
                    'currency' => $order->get_currency(),
                    'value' => (float) $order->get_total(),
                    'contents' => $contents,
                ],
                'user_data' => $user_data,
            ];

            $payload = [
                'data' => [$event],
            ];

            // Build URL với access_token + test_event_code (nếu có)
            $query_args = [
                'access_token' => $access_token,
            ];

            if (!empty($options['test_event_code'])) {
                $query_args['test_event_code'] = $options['test_event_code'];
            }

            $url = add_query_arg(
                $query_args,
                sprintf(
                    'https://graph.facebook.com/v20.0/%s/events',
                    rawurlencode($dataset_id)
                )
            );

            $response = wp_remote_post(
                $url,
                [
                    'headers' => [
                        'Content-Type' => 'application/json',
                    ],
                    'body' => wp_json_encode($payload),
                    'timeout' => 20,
                ]
            );

            if (is_wp_error($response)) {
                $this->log('Meta API error: ' . $response->get_error_message(), $options);
            } else {
                $body = wp_remote_retrieve_body($response);
                $this->log('Meta API response for order ' . $order_id . ': ' . $body, $options);
            }
        }

        /**
         * Ghi log nếu bật
         */
        private function log($message, $options)
        {
            if (!empty($options['enable_logging'])) {
                if (defined('WP_DEBUG') && WP_DEBUG && defined('WP_DEBUG_LOG') && WP_DEBUG_LOG) {
                    error_log('[LHT Meta Dataset] ' . $message);
                }
            }
        }
    }

endif;

/**
 * Khởi tạo plugin sau khi plugins load
 */
function ht_meta_dataset_tracker_init()
{
    if (class_exists('HT_Meta_Dataset_Tracker')) {
        new HT_Meta_Dataset_Tracker();
    }
}
add_action('plugins_loaded', 'ht_meta_dataset_tracker_init');

/**
 * Hook kích hoạt plugin
 */
register_activation_hook(__FILE__, ['HT_Meta_Dataset_Tracker', 'activate']);
