<?php
/**
 * Plugin Name: LHT Sepay for Funnels
 * Description: Tích hợp cổng Sepay với các Plugin Funnel
 * Author: Lê Hồng Tuân
 * Author URI: https://lehongtuan.com
 * Version: 1.0.1
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Tự động cập nhật plugin (self-hosted) với Plugin Update Checker
 * - Yêu cầu thư mục: plugin-update-checker/ nằm cùng cấp file plugin này
 * - File JSON update: https://plugin-update.lehongtuan.com/lht-sepay-for-funnels/lht-sepay-for-funnels.json
 */
if ( ! class_exists( '\YahnisElsts\PluginUpdateChecker\v5\PucFactory' ) ) {
    require plugin_dir_path( __FILE__ ) . 'plugin-update-checker/plugin-update-checker.php';
}

use YahnisElsts\PluginUpdateChecker\v5\PucFactory;

$sepay_update_checker = PucFactory::buildUpdateChecker(
    'https://plugin-update.lehongtuan.com/lht-sepay-for-funnels/lht-sepay-for-funnels.json', // URL JSON metadata
    __FILE__,                                                                                // File chính của plugin
    'lht-sepay-for-funnels'                                                                  // Slug plugin (trùng folder)
);

/**
 * Kích hoạt plugin: tạo API key ngẫu nhiên nếu chưa có
 */
register_activation_hook(__FILE__, 'setup_api_key_on_activation');

function setup_api_key_on_activation() {
    if ( ! get_option('api_key') ) {
        $api_key = generateRandomString(32);
        update_option('api_key', $api_key);
    }
}

/**
 * Hàm sinh chuỗi random cho API key (PHP)
 */
function generateRandomString($length) {
    $characters = 'ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789';
    $result     = '';

    for ($i = 0; $i < $length; $i++) {
        $result .= $characters[rand(0, strlen($characters) - 1)];
    }
    return $result;
}

/**
 * AJAX check trạng thái đơn hàng – để tự reload trang thankyou
 */
add_action('wp_ajax_check_order_status', 'check_order_status');
add_action('wp_ajax_nopriv_check_order_status', 'check_order_status');

function check_order_status() {
    if (isset($_POST['order_id'])) {
        $order_id = intval($_POST['order_id']);
        $order    = wc_get_order($order_id);

        if ($order) {
            $shipping_total = $order->get_shipping_total();
            $has_shipping   = $shipping_total > 0;

            // Có phí ship => cần processing
            if ($has_shipping && $order->get_status() === 'processing') {
                wp_send_json_success();
            }
            // Không phí ship => cần completed
            elseif (!$has_shipping && $order->get_status() === 'completed') {
                wp_send_json_success();
            } else {
                wp_send_json_error();
            }
        }
    }
    wp_send_json_error();
}

/**
 * REST API Webhook từ Sepay
 */
add_action('rest_api_init', function () {
    register_rest_route('pay/v1', '/webhook', array(
        'methods'             => 'POST',
        'callback'            => 'handle_pay_webhook',
        'permission_callback' => '__return_true',
    ));
});

/**
 * Xử lý Webhook Sepay
 */
function handle_pay_webhook(WP_REST_Request $request) {
    $api_key     = get_option('api_key');
    $auth_header = $request->get_header('Authorization');

    // Kiểm tra Authorization header
    if ($auth_header !== 'Apikey ' . $api_key) {
        return new WP_REST_Response(['message' => 'Unauthorized'], 401);
    }

    $data = $request->get_json_params();

    if (empty($data)) {
        return new WP_REST_Response(['message' => 'Không thể giải mã dữ liệu JSON.'], 400);
    }

    $code           = isset($data['code']) ? trim($data['code']) : null;
    $transferAmount = isset($data['transferAmount']) ? trim($data['transferAmount']) : null;
    $gateway        = isset($data['gateway']) ? trim($data['gateway']) : null;
    $transferType   = isset($data['transferType']) ? trim($data['transferType']) : null;

    if (is_null($code)) {
        return new WP_REST_Response(['message' => 'Thiếu trường "code".'], 400);
    }
    if (is_null($transferAmount)) {
        return new WP_REST_Response(['message' => 'Thiếu trường "transferAmount".'], 400);
    }
    if (is_null($gateway)) {
        return new WP_REST_Response(['message' => 'Thiếu trường "gateway".'], 400);
    }
    if (is_null($transferType)) {
        return new WP_REST_Response(['message' => 'Thiếu trường "transferType".'], 400);
    }

    // Chỉ xử lý giao dịch tiền vào
    if ($transferType !== 'in') {
        return new WP_REST_Response(['message' => 'Không phải tiền vào.'], 400);
    }

    // Lấy ID đơn hàng từ code: noi_dung + order_id
    $noi_dung = get_option('noi_dung');
    $order_id = intval(str_replace($noi_dung, '', $code));

    $order = wc_get_order($order_id);

    if ($order) {
        $order_total     = $order->get_total();
        $received_amount = floatval($transferAmount);

        if ($received_amount == $order_total) {
            $shipping_total = $order->get_shipping_total();

            if ($shipping_total > 0) {
                $order->update_status(
                    'processing',
                    'Đã nhận số tiền ' . number_format($received_amount, 0, ',', '.') . ' ₫ qua tài khoản ' . esc_html($gateway) . ' vào lúc ' . current_time('mysql')
                );
            } else {
                $order->update_status(
                    'completed',
                    'Đã nhận số tiền ' . number_format($received_amount, 0, ',', '.') . ' ₫ qua tài khoản ' . esc_html($gateway) . ' vào lúc ' . current_time('mysql')
                );
            }

            return new WP_REST_Response(['message' => 'Đơn hàng đã được cập nhật'], 200);
        } else {
            $order->add_order_note(
                'Số tiền chuyển ' . number_format($received_amount, 0, ',', '.') . ' ₫ không khớp với số tiền của đơn hàng ' . number_format($order_total, 0, ',', '.') . ' ₫.'
            );
            return new WP_REST_Response([
                'message' => 'Số tiền chuyển không khớp với số tiền của đơn hàng. Ghi chú đã được thêm.'
            ], 200);
        }
    } else {
        return new WP_REST_Response(['message' => 'Không tìm thấy đơn hàng với ID: ' . $order_id], 404);
    }
}

/**
 * Menu Cài đặt trong Settings
 */
add_action("admin_menu", "cai_dat_apibank_menu");
function cai_dat_apibank_menu() {
    add_submenu_page(
        'options-general.php',
        'Cài đặt LHT Sepay for Funnels',
        'LHT Sepay for Funnels',
        'manage_options',
        'setup_cong_thanh_toan',
        'apibank_management_plugin_settings_page'
    );

    add_action('admin_init', 'register_apibank_plugin_settings');
}

/**
 * Enqueue CSS giao diện (frontend)
 */
function cong_thanh_toan_sms() {
    wp_enqueue_style('cong-thanh-toan-sms', plugins_url('styles.css', __FILE__));
}
add_action('wp_enqueue_scripts', 'cong_thanh_toan_sms');

/**
 * Hiển thị hướng dẫn thanh toán trên trang thankyou WooCommerce
 */
add_action('woocommerce_order_details_before_order_table', 'huongDanThanhToan', 10, 1);
function huongDanThanhToan($order) {
    if (($order->get_status() == 'completed' || $order->get_status() == 'processing') && $order->get_payment_method() == 'bacs') {
        ?>
        <div class="order-completed" style="margin: 20px; text-align: center;">
            <div
                style=" display: inline-block; border: 1px solid #E2E2E2; border-radius: 10px; padding: 20px; text-align: center; max-width: 600px; background: #fff">
                <div style="font-weight: bold;">THANH TOÁN THÀNH CÔNG</div>
                <span style="display: block">(Hệ thống đã nhận được thanh toán hãy vào Email của bạn để nhận khoá học)</span>
                <img style="width: 100px; display: block; margin:auto"
                    src="<?php echo esc_url(plugins_url('imgs/', __FILE__) . 'thanh-toan.svg'); ?>">
            </div>
        </div>
        <?php
    } elseif (($order->get_status() == 'on-hold') && ($order->get_payment_method() == "bacs")) {
        ?>
        <p id="thongBaoCopy" style="display:none;">Sao Chép Thành Công!</p>
        <section style="display: flex;">
            <div id="thongTinThanhToanHH">
                <div id="cot1HH">
                    <span><b>Bước 1:</b> Mở Ví điện tử/Ngân hàng</span>
                    <div class="iconCenterHH">
                        <span><b>Bước 2:</b> Chọn</span>
                        <img src="<?php echo esc_url(plugins_url('imgs/', __FILE__) . 'qr.svg'); ?>">
                        <span>và quét mã</span>
                    </div>
                    <img class="qrVietqr"
                        src="https://api.vietqr.io/<?php echo esc_attr(get_option('ngan_hang')) ?>/<?php echo esc_attr(get_option('so_tai_khoan')); ?>/<?php echo $order->get_total(); ?>/<?php echo rawurlencode(" ".get_option('noi_dung').$order->get_id()." "); ?>/qr_only.jpg">
                    <img class="napas247" width="120" src="<?php echo esc_url(plugins_url('imgs/', __FILE__) . 'napas247.png'); ?>">

                    <span><b>Bước 3:</b> Xác Nhận Chuyển Khoản</span>
                </div>
                <div style="text-align: center" id="cot2HH">
                    <div class="anMoblie">
                        <div style="font-weight: bold">
                            <span style="color: red">Hoặc:</span> CHUYỂN KHOẢN THEO THÔNG TIN
                        </div>
                        <div style="font-size:small; line-height: 1.2em">Sử dụng Ví điện tử MoMo/ZaloPay<br>Hoặc ứng dụng ngân hàng để
                            chuyển khoản nhanh 247
                        </div>
                    </div>

                    <div class="anPc">
                        <div style="margin:auto">
                            <span style="color: red; font-weight: bold">Cách 1:</span> <b>QUÉT MÃ THANH TOÁN</b>
                        </div>
                        <img class="qrVietqr"
                            src="https://api.vietqr.io/<?php echo esc_attr(get_option('ngan_hang')) ?>/<?php echo esc_attr(get_option('so_tai_khoan')); ?>/<?php echo $order->get_total(); ?>/<?php echo rawurlencode(" ".get_option('noi_dung').$order->get_id()." "); ?>/qr_only.jpg">
                        <span><b>Bước 1:</b> Giữ tay vào mã QR code</span>
                        <span>sau đó nhấn lưu ảnh vào Album (Photos)</span>
                        <span><b>Bước 2:</b> Mở Ví điện tử/Ngân hàng</span>
                        <div class="iconCenterHH">
                            <span><b>Bước 3:</b> Chọn</span>
                            <img src="<?php echo esc_url(plugins_url('imgs/', __FILE__) . 'qr.svg'); ?>">
                            <span>và chọn ảnh Qr mới tải về</span>
                        </div>
                        <span><b>Bước 4:</b> Xác Nhận Chuyển Khoản</span>
                        <div style="border: 1px dashed #e2e2e2; width:100%; border-width: 0 0 1px 0;"><br>
                        </div>
                        <div style="padding-top: 20px; margin:auto;">
                            <span style="color: red; font-weight:bold;">Cách 2:</span> <b>CHUYỂN KHOẢN THỦ CÔNG</b>
                        </div>
                    </div>

                    <div>
                        <div style="font-weight:bold; font-size: large"><?php echo esc_html(get_option('chu_tai_khoan')); ?></div>
                        <table>
                            <tbody>
                                <tr>
                                    <th>Ngân Hàng:</th>
                                    <th>
                                        <img
                                            src="<?php echo esc_url(plugins_url('imgs/', __FILE__) . esc_attr(get_option('ngan_hang')) . '.svg'); ?>">
                                    </th>
                                </tr>
                                <tr>
                                    <th>Tài Khoản:</th>
                                    <th>
                                        <span id="stk"><?php echo esc_html(get_option('so_tai_khoan')); ?></span>
                                        <button type="button" onclick="copy('stk')">
                                            <img width="20"
                                                src="<?php echo esc_url(plugins_url('imgs/', __FILE__) . 'copy.png'); ?>"
                                                alt="">
                                        </button>
                                    </th>
                                </tr>
                                <tr>
                                    <th>Số Tiền:</th>
                                    <th>
                                        <?php echo number_format($order->get_total(),0,",","."); ?> ₫
                                        <span style="display:none" id="st"><?php echo esc_html($order->get_total()); ?></span>
                                        <button type="button" onclick="copy('st')">
                                            <img width="20"
                                                src="<?php echo esc_url(plugins_url('imgs/', __FILE__) . 'copy.png'); ?>"
                                                alt="">
                                        </button>
                                    </th>
                                </tr>
                                <tr>
                                    <th>Nội Dung<span style="color: red">*</span>:</th>
                                    <th>
                                        <span class="noiDungCopy"
                                            id="nd"><?php echo esc_html(get_option('noi_dung') . $order->get_id()); ?></span>
                                        <button type="button" onclick="copy('nd')">
                                            <img width="20"
                                                src="<?php echo esc_url(plugins_url('imgs/', __FILE__) . 'copy.png'); ?>"
                                                alt="">
                                        </button>
                                    </th>
                                </tr>
                            </tbody>
                        </table>
                    </div>
                    <div>
                        <div>
                            <span style="color: red">*</span>Phải nhập <b>chính xác</b>
                            <span style="color: red">"Nội Dung"</span>.
                        </div>
                        <div style="background: #f3f3f3; padding: 5px; margin:0; border-radius: 5px">
                            Để đơn hàng xác nhận tự động.
                        </div>
                        <img src="<?php echo esc_url(plugins_url('imgs/loading.gif', __FILE__)); ?>" alt="Đang tải" style="width: 20px; vertical-align: middle; margin-left: 5px;">
                    </div>
                </div>
            </div>
        </section>

        <div style="display:flex">
            <span style="margin:auto">Mã QR - VietQr hỗ trợ cho tất cả các Ví điện tử và tất cả các Ứng dụng chuyển khoản
                nhanh 247 của Ngân hàng Việt Nam.</span>
        </div>
        <script>
        jQuery(document).ready(function($) {
            let timer = null;
            const is_completed = $('.order-completed').length;
            if (!is_completed) {
                timer = setInterval(() => {
                    $.ajax({
                        url: '<?php echo admin_url('admin-ajax.php'); ?>',
                        type: 'POST',
                        dataType: 'json',
                        data: {
                            action: 'check_order_status',
                            order_id: '<?php echo $order->get_id(); ?>',
                        },
                    })
                    .done(function(res) {
                        if (res.success) {
                            clearInterval(timer);
                            window.location.reload();
                        }
                    });
                }, 3000);
            }
        });

        function copy(copyId) {
            let inputElement = document.createElement("input");
            inputElement.type = "text";
            let copyText = document.getElementById(copyId).innerHTML;
            inputElement.value = copyText;
            document.body.appendChild(inputElement);
            inputElement.select();
            document.execCommand('copy');
            document.body.removeChild(inputElement);

            document.getElementById("thongBaoCopy").style.display = "flex";
            setTimeout(function() {
                document.getElementById("thongBaoCopy").style.display = "none";
            }, 2000);
        }
        </script>
        <?php
    }
}

/**
 * Đăng ký options
 */
function register_apibank_plugin_settings() {
    register_setting('tuy_chon_apibank', 'api_sms');
    register_setting('tuy_chon_apibank', 'noi_dung');
    register_setting('tuy_chon_apibank', 'so_tai_khoan');
    register_setting('tuy_chon_apibank', 'chu_tai_khoan');
    register_setting('tuy_chon_apibank', 'ngan_hang');
    register_setting('tuy_chon_apibank', 'api_key');
}

/**
 * CSS cho trang cài đặt admin (card layout)
 */
add_action('admin_head', 'lht_sepay_admin_css');
function lht_sepay_admin_css() {
    $screen = get_current_screen();
    if ( ! $screen || $screen->id !== 'settings_page_setup_cong_thanh_toan' ) {
        return;
    }
    ?>
    <style>
        .lht-sepay-wrap {
            max-width: 900px;
        }

        .lht-sepay-header {
            margin-bottom: 20px;
        }

        .lht-sepay-title {
            font-size: 24px;
            font-weight: 600;
            margin-bottom: 5px;
        }

        .lht-sepay-subtitle {
            color: #555d66;
            font-size: 13px;
            max-width: 640px;
        }

        .lht-sepay-card {
            background: #fff;
            border-radius: 8px;
            border: 1px solid #e5e5e5;
            box-shadow: 0 1px 3px rgba(0,0,0,0.04);
            padding: 24px 24px 16px;
            margin-bottom: 16px;
        }

        .lht-sepay-card h2 {
            margin-top: 0;
            font-size: 16px;
            margin-bottom: 16px;
        }

        .lht-sepay-badge {
            display: inline-block;
            padding: 2px 8px;
            border-radius: 999px;
            background: #f3f4f6;
            font-size: 11px;
            font-weight: 500;
            color: #555d66;
            margin-left: 8px;
        }

        .lht-sepay-wrap .form-table {
            margin-top: 0;
        }

        .lht-sepay-wrap .form-table th {
            width: 200px;
            padding: 10px 10px 10px 0;
        }

        .lht-sepay-wrap .form-table td {
            padding: 10px 0;
        }

        .lht-sepay-wrap input[type="text"],
        .lht-sepay-wrap select {
            max-width: 100%;
        }

        .lht-sepay-url-field {
            font-family: Menlo, Monaco, Consolas, "Courier New", monospace;
        }

        .lht-sepay-help {
            font-size: 12px;
            color: #6c757d;
            margin-top: 4px;
            display: block;
        }

        .lht-sepay-actions {
            margin-top: 10px;
        }

        .lht-sepay-actions .button {
            margin-right: 6px;
        }

        .lht-sepay-footer-note {
            margin-top: 10px;
            font-size: 12px;
            color: #6c757d;
        }
    </style>
    <?php
}

/**
 * Trang cài đặt admin – card layout + JS API key
 */
function apibank_management_plugin_settings_page() {
    $banks = [
        ['id' => 970436, 'label' => 'Vietcombank'],
        ['id' => 970416, 'label' => 'ACB'],
        ['id' => 970407, 'label' => 'Techcombank'],
        ['id' => 970426, 'label' => 'MSB'],
        ['id' => 970422, 'label' => 'MB Bank'],
        ['id' => 970406, 'label' => 'Đông Á'],
        ['id' => 970432, 'label' => 'VP Bank'],
        ['id' => 970428, 'label' => 'Nam Á'],
        ['id' => 970418, 'label' => 'BIDV'],
        ['id' => 970403, 'label' => 'Sacombank'],
        ['id' => 970441, 'label' => 'VIB'],
        ['id' => 970415, 'label' => 'ViettinBank'],
        ['id' => 970423, 'label' => 'TP Bank'],
        ['id' => 970431, 'label' => 'Eximbank'],
        ['id' => 970452, 'label' => 'KienlongBank'],
    ];

    $ngan_hang = get_option('ngan_hang');
    ?>
    <div class="wrap lht-sepay-wrap">
        <div class="lht-sepay-header">
            <div class="lht-sepay-title">
                LHT Sepay for Funnels
                <span class="lht-sepay-badge">v1.0.1</span>
            </div>
            <p class="lht-sepay-subtitle">
                Tự động đối soát thanh toán chuyển khoản ngân hàng (VietQR) cho WooCommerce, cập nhật trạng thái đơn
                hàng khi nhận được Webhook từ Sepay. Cấu hình tài khoản ngân hàng, nội dung chuyển khoản và API key tại đây.
            </p>
        </div>

        <form method="post" action="options.php">
            <?php settings_fields('tuy_chon_apibank'); ?>
            <?php do_settings_sections('tuy_chon_apibank'); ?>

            <div class="lht-sepay-card">
                <h2>Thông tin tài khoản nhận tiền</h2>
                <table class="form-table">
                    <tbody>
                        <tr>
                            <th><label for="ngan_hang">Ngân hàng</label></th>
                            <td>
                                <select name="ngan_hang" id="ngan_hang">
                                    <option value="" disabled <?php echo empty($ngan_hang) ? 'selected' : ''; ?>>Chọn Ngân Hàng</option>
                                    <?php foreach ($banks as $b) : ?>
                                        <option value="<?php echo esc_attr($b['id']); ?>" <?php selected($ngan_hang, $b['id']); ?>>
                                            <?php echo esc_html($b['label']); ?>
                                        </option>
                                    <?php endforeach; ?>
                                </select>
                                <span class="lht-sepay-help">
                                    Mã ngân hàng dùng cho VietQR. Plugin sẽ tự tạo link QR dựa trên ngân hàng này.
                                </span>
                            </td>
                        </tr>
                        <tr>
                            <th><label for="chu_tai_khoan">Chủ tài khoản</label></th>
                            <td>
                                <input type="text" name="chu_tai_khoan" id="chu_tai_khoan"
                                    value="<?php echo esc_attr(get_option('chu_tai_khoan')); ?>" />
                                <span class="lht-sepay-help">
                                    Tên chủ tài khoản hiển thị cho khách khi thanh toán.
                                </span>
                            </td>
                        </tr>
                        <tr>
                            <th><label for="so_tai_khoan">Số tài khoản</label></th>
                            <td>
                                <input type="text" name="so_tai_khoan" id="so_tai_khoan"
                                    value="<?php echo esc_attr(get_option('so_tai_khoan')); ?>" />
                                <span class="lht-sepay-help">
                                    Số tài khoản nhận tiền, dùng trong QR và hướng dẫn chuyển khoản.
                                </span>
                            </td>
                        </tr>
                        <tr>
                            <th><label for="noi_dung">Nội dung thanh toán</label></th>
                            <td>
                                <input type="text" name="noi_dung" id="noi_dung"
                                    value="<?php echo esc_attr(get_option('noi_dung')); ?>" />
                                <span class="lht-sepay-help">
                                    Prefix nội dung chuyển khoản, ví dụ: <code>DONHANG_</code>. Plugin sẽ tự cộng thêm ID đơn:
                                    <code>DONHANG_1234</code>.
                                </span>
                            </td>
                        </tr>
                    </tbody>
                </table>
            </div>

            <div class="lht-sepay-card">
                <h2>API key &amp; Webhook</h2>
                <table class="form-table">
                    <tbody>
                        <tr>
                            <th><label for="api_key">API key</label></th>
                            <td>
                                <input type="text" name="api_key" id="api_key" class="lht-sepay-url-field"
                                    readonly style="width: 100%; padding: 10px;"
                                    value="<?php echo esc_attr(get_option('api_key')); ?>" />
                                <div class="lht-sepay-actions">
                                    <button type="button" id="generate_api_key" class="button">Tạo ngẫu nhiên</button>
                                    <button type="button" id="copy_api_key" class="button">Sao chép</button>
                                </div>
                                <span class="lht-sepay-help">
                                    API key dùng để xác thực Webhook từ Sepay. Mỗi site nên dùng một API key riêng.
                                </span>
                            </td>
                        </tr>
                        <tr>
                            <th>URL nhận Webhook</th>
                            <td>
                                <input type="text"
                                    value="<?php echo esc_url(rest_url('pay/v1/webhook')); ?>"
                                    readonly
                                    class="lht-sepay-url-field"
                                    style="width: 100%; padding: 10px;">
                                <span class="lht-sepay-help">
                                    Dán URL này vào phần Webhook của Sepay. Yêu cầu header:
                                    <code>Authorization: Apikey &lt;API_KEY&gt;</code>.
                                </span>
                            </td>
                        </tr>
                    </tbody>
                </table>
                <p class="lht-sepay-footer-note">
                    Trạng thái đơn hàng sẽ tự động chuyển sang <strong>processing</strong> (có phí ship) hoặc
                    <strong>completed</strong> (không phí ship) khi số tiền nhận khớp với tổng đơn.
                </p>
            </div>

            <?php submit_button(); ?>
        </form>

        <script>
        document.getElementById('generate_api_key').addEventListener('click', function() {
            var apiKey = generateRandomString(32);
            document.getElementById('api_key').value = apiKey;
        });

        document.getElementById('copy_api_key').addEventListener('click', function() {
            var apiKeyField = document.getElementById('api_key');
            apiKeyField.select();
            document.execCommand('copy');
            alert('Đã sao chép API key vào clipboard!');
        });

        function generateRandomString(length) {
            var characters = 'ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789';
            var result = '';
            for (var i = 0; i < length; i++) {
                result += characters.charAt(Math.floor(Math.random() * characters.length));
            }
            return result;
        }
        </script>
    </div>
    <?php
}

/**
 * Gửi thông tin ngân hàng trong email WooCommerce
 */
add_filter('woocommerce_bacs_accounts', '__return_false');
add_action('woocommerce_email_before_order_table', 'email_instructions_HH', 10, 3);

function email_instructions_HH($order, $sent_to_admin, $plain_text = false) {
    if ( ! $sent_to_admin && 'bacs' === $order->get_payment_method() && $order->has_status('on-hold') ) {
        bank_details_HH($order->get_id());
    }
}

function thankyou_page_HH($order_id) {
    bank_details_HH($order_id);
}

function bank_details_HH($order_id = '') {
    $noi_dung      = esc_attr(get_option('noi_dung'));
    $so_tai_khoan  = esc_attr(get_option('so_tai_khoan'));
    $chu_tai_khoan = esc_attr(get_option('chu_tai_khoan'));
    $ngan_hang     = esc_attr(get_option('ngan_hang'));

    if ($ngan_hang==970436) {$bank='Vietcombank';}
    if ($ngan_hang==970416) {$bank='ACB';}
    if ($ngan_hang==970407) {$bank='Techcombank';}
    if ($ngan_hang==970426) {$bank='MSB';}
    if ($ngan_hang==970422) {$bank='MB Bank';}
    if ($ngan_hang==970406) {$bank='Đông Á';}
    if ($ngan_hang==970432) {$bank='VP Bank';}
    if ($ngan_hang==970428) {$bank='Nam Á';}
    if ($ngan_hang==970418) {$bank='BIDV';}
    if ($ngan_hang==970403) {$bank='Sacombank';}
    if ($ngan_hang==970441) {$bank='VIB';}
    if ($ngan_hang==970415) {$bank='ViettinBank';}
    if ($ngan_hang==970423) {$bank='TP Bank';}
    if ($ngan_hang==970431) {$bank='Eximbank';}
    if ($ngan_hang==970452) {$bank='KienlongBank';}
    ?>
    <h2>Thông tin chuyển khoản ngân hàng</h2>
    <ul>
        <li>Ngân hàng: <?php echo esc_html($bank); ?></li>
        <li>Số tài khoản: <?php echo esc_html($so_tai_khoan); ?></li>
        <li>Chủ tài khoản: <?php echo esc_html($chu_tai_khoan); ?></li>
        <li>Nội dung chuyển khoản: <?php echo esc_html($noi_dung . $order_id); ?></li>
    </ul>
    <?php
}
