<?php
/**
 * Plugin Name: LHT Short Links
 * Plugin URI: https://lehongtuan.com
 * Description: Rút gọn link dạng https://domain.com/code, có thể sửa/xoá code, đếm click, redirect trang chủ, tìm kiếm & phân trang, và tuỳ chọn xóa sạch dữ liệu khi gỡ plugin.
 * Author: Lê Hồng Tuân
 * Author URI: https://lehongtuan.com
 * Version: 1.0.1
 */

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Tự động cập nhật plugin (self-hosted) với Plugin Update Checker
 * - Yêu cầu thư mục: plugin-update-checker/ nằm cùng cấp file plugin này
 * - File JSON update: https://lehongtuan.com/updates/lht-short-links.json
 */
if (!class_exists('\YahnisElsts\PluginUpdateChecker\v5\PucFactory')) {
    // Đường dẫn tới thư viện update checker
    require plugin_dir_path(__FILE__) . 'plugin-update-checker/plugin-update-checker.php';
}

use YahnisElsts\PluginUpdateChecker\v5\PucFactory;

$ht_sl_update_checker = PucFactory::buildUpdateChecker(
    'https://plugin-update.lehongtuan.com/lht-short-links/lht-short-links.json',
    __FILE__,
    'lht-short-links'
);


class LHT_Short_Links
{

    const TABLE = 'lht_short_links';
    const OPTION_HOMEPAGE_REDIRECT = 'lht_sl_homepage_redirect';
    const OPTION_DELETE_ON_UNINSTALL = 'lht_sl_delete_on_uninstall';

    public function __construct()
    {
        // Tạo bảng khi kích hoạt plugin
        register_activation_hook(__FILE__, [$this, 'activate']);

        // Admin page + xử lý form shortlink + xoá shortlink
        add_action('admin_menu', [$this, 'admin_menu']);
        add_action('admin_post_lht_sl_save', [$this, 'handle_save']);
        add_action('admin_post_lht_sl_delete', [$this, 'handle_delete']);

        // Đăng ký settings cho options.php
        add_action('admin_init', [$this, 'register_settings']);

        // Rewrite rule + redirect xử lý shortlink
        add_action('init', [$this, 'add_rewrite']);
        add_filter('query_vars', [$this, 'register_query_var']);
        add_action('template_redirect', [$this, 'handle_redirect']);
    }

    /** Tạo bảng khi kích hoạt plugin */
    public function activate()
    {
        global $wpdb;
        $table = $wpdb->prefix . self::TABLE;
        $charset = $wpdb->get_charset_collate();

        $sql = "CREATE TABLE IF NOT EXISTS $table (
            id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
            code VARCHAR(191) NOT NULL UNIQUE,
            long_url TEXT NOT NULL,
            click_count BIGINT UNSIGNED NOT NULL DEFAULT 0,
            created_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP
        ) $charset;";

        require_once ABSPATH . 'wp-admin/includes/upgrade.php';
        dbDelta($sql);

        // Đăng ký rewrite và flush
        $this->add_rewrite();
        flush_rewrite_rules();
    }

    /** Rewrite: /code → shortlink, tránh đụng wp-admin, wp-json, v.v. */
    public function add_rewrite()
    {
        // Bất kỳ segment 1 cấp (không chứa slash), trừ các prefix hệ thống
        add_rewrite_rule(
            '^(?!wp-admin|wp-content|wp-includes|wp-json|feed|index\.php|xmlrpc\.php)([^/]+)/?$',
            'index.php?lht_sl_code=$matches[1]',
            'top'
        );
    }

    public function register_query_var($vars)
    {
        $vars[] = 'lht_sl_code';
        return $vars;
    }

    /** Đăng ký settings cho phần options */
    public function register_settings()
    {
        register_setting('lht_sl_settings', self::OPTION_HOMEPAGE_REDIRECT);
        register_setting('lht_sl_settings', self::OPTION_DELETE_ON_UNINSTALL);
    }

    /** Tạo menu admin */
    public function admin_menu()
    {
        add_menu_page(
            'LHT Short Links',
            'Short Links',
            'manage_options',
            'lht-short-links',
            [$this, 'render_admin_page'],
            'dashicons-admin-links',
            80
        );
    }

    /** Trang quản lý + tạo/sửa shortlink + danh sách + settings */
    public function render_admin_page()
    {
        if (!current_user_can('manage_options'))
            return;

        global $wpdb;
        $table = $wpdb->prefix . self::TABLE;

        // Nếu có ?edit_id= thì load dữ liệu để sửa
        $edit_id = isset($_GET['edit_id']) ? intval($_GET['edit_id']) : 0;
        $edit_row = null;
        if ($edit_id > 0) {
            $edit_row = $wpdb->get_row(
                $wpdb->prepare("SELECT * FROM $table WHERE id = %d", $edit_id)
            );
        }

        // Tìm kiếm
        $search = isset($_GET['s']) ? sanitize_text_field($_GET['s']) : '';

        // Phân trang
        $per_page = 20;
        $paged = isset($_GET['paged']) ? max(1, intval($_GET['paged'])) : 1;
        $offset = ($paged - 1) * $per_page;

        // Điều kiện WHERE cho search
        $where = '';
        $where_args = [];
        if ($search !== '') {
            $where = "WHERE code LIKE %s OR long_url LIKE %s";
            $like = '%' . $wpdb->esc_like($search) . '%';
            $where_args = [$like, $like];
        }

        // Tổng số record
        if ($where) {
            $total = $wpdb->get_var(
                $wpdb->prepare("SELECT COUNT(*) FROM $table $where", $where_args)
            );
        } else {
            $total = $wpdb->get_var("SELECT COUNT(*) FROM $table");
        }

        $total_pages = $total > 0 ? ceil($total / $per_page) : 1;

        // Lấy danh sách shortlink theo trang
        if ($where) {
            $sql = $wpdb->prepare(
                "SELECT * FROM $table $where ORDER BY id DESC LIMIT %d OFFSET %d",
                array_merge($where_args, [$per_page, $offset])
            );
        } else {
            $sql = $wpdb->prepare(
                "SELECT * FROM $table ORDER BY id DESC LIMIT %d OFFSET %d",
                $per_page,
                $offset
            );
        }
        $links = $wpdb->get_results($sql);

        $homepage_redirect = get_option(self::OPTION_HOMEPAGE_REDIRECT, '');
        $delete_on_uninstall = (bool) get_option(self::OPTION_DELETE_ON_UNINSTALL, 0);

        // Link phân trang
        $base = remove_query_arg('paged', $_SERVER['REQUEST_URI']);
        $base = add_query_arg('paged', '%#%', $base);
        $pagination = paginate_links([
            'base' => esc_url($base),
            'format' => '',
            'current' => $paged,
            'total' => max(1, $total_pages),
            'prev_text' => '&laquo;',
            'next_text' => '&raquo;',
            'type' => 'list',
        ]);
        ?>
        <div class="wrap">
            <h1>LHT Short Links</h1>

            <!-- Form tạo/sửa shortlink -->
            <h2><?php echo $edit_row ? 'Sửa short link' : 'Thêm short link mới'; ?></h2>

            <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>">
                <?php wp_nonce_field('lht_sl_save', 'lht_sl_nonce'); ?>
                <input type="hidden" name="action" value="lht_sl_save">

                <?php if ($edit_row): ?>
                    <input type="hidden" name="id" value="<?php echo esc_attr($edit_row->id); ?>">
                <?php endif; ?>

                <table class="form-table">
                    <tr>
                        <th><label for="lht_sl_long_url">URL gốc</label></th>
                        <td>
                            <input name="long_url" id="lht_sl_long_url" type="text" class="regular-text" required
                                value="<?php echo esc_attr($edit_row ? $edit_row->long_url : ''); ?>">
                        </td>
                    </tr>
                    <tr>
                        <th><label for="lht_sl_code">Short code</label></th>
                        <td>
                            <input name="code" id="lht_sl_code" type="text" class="regular-text"
                                placeholder="abc123 (để trống = random)"
                                value="<?php echo esc_attr($edit_row ? $edit_row->code : ''); ?>">
                            <p class="description">
                                Short URL sẽ là:
                                <code><?php echo esc_html(trailingslashit(home_url('/'))); ?><strong>{code}</strong></code>
                            </p>
                        </td>
                    </tr>
                </table>

                <p class="submit">
                    <button class="button button-primary">
                        <?php echo $edit_row ? 'Cập nhật short link' : 'Tạo short link'; ?>
                    </button>
                    <?php if ($edit_row): ?>
                        <a href="<?php echo esc_url(admin_url('admin.php?page=lht-short-links')); ?>" class="button">Hủy sửa</a>
                    <?php endif; ?>
                </p>
            </form>

            <hr>

            <!-- Tìm kiếm + danh sách shortlink (ngay sau form) -->
            <h2>Danh sách shortlink</h2>

            <form method="get" style="margin-bottom: 10px;">
                <input type="hidden" name="page" value="lht-short-links">
                <label for="lht_sl_search">Tìm kiếm: </label>
                <input type="search" id="lht_sl_search" name="s" value="<?php echo esc_attr($search); ?>"
                    placeholder="Tìm theo code hoặc URL..." />
                <button class="button">Tìm</button>
                <?php if ($search !== ''): ?>
                    <a class="button" href="<?php echo esc_url(admin_url('admin.php?page=lht-short-links')); ?>">Xoá lọc</a>
                <?php endif; ?>
            </form>

            <p>
                Tổng cộng: <strong><?php echo intval($total); ?></strong> shortlink.
                <?php if ($total_pages > 1): ?>
                    — Trang <?php echo intval($paged); ?>/<?php echo intval($total_pages); ?>
                <?php endif; ?>
            </p>

            <table class="widefat fixed striped">
                <thead>
                    <tr>
                        <th width="4%">ID</th>
                        <th width="12%">Code</th>
                        <th width="20%">Short URL</th>
                        <th>URL gốc</th>
                        <th width="8%">Click</th>
                        <th width="15%">Ngày tạo</th>
                        <th width="12%">Hành động</th>
                    </tr>
                </thead>
                <tbody>
                    <?php if ($links):
                        foreach ($links as $r): ?>
                            <tr>
                                <td><?php echo intval($r->id); ?></td>
                                <td><code><?php echo esc_html($r->code); ?></code></td>
                                <td>
                                    <a href="<?php echo esc_url(home_url('/' . $r->code)); ?>" target="_blank">
                                        <?php echo esc_html(home_url('/' . $r->code)); ?>
                                    </a>
                                </td>
                                <td style="word-break:break-all;"><?php echo esc_html($r->long_url); ?></td>
                                <td><?php echo intval($r->click_count); ?></td>
                                <td><?php echo esc_html($r->created_at); ?></td>
                                <td>
                                    <a class="button button-small" href="<?php echo esc_url(add_query_arg(
                                        [
                                            'page' => 'lht-short-links',
                                            'edit_id' => $r->id,
                                        ],
                                        admin_url('admin.php')
                                    )); ?>">
                                        Sửa
                                    </a>

                                    <?php
                                    $delete_url = wp_nonce_url(
                                        add_query_arg([
                                            'action' => 'lht_sl_delete',
                                            'id' => $r->id,
                                        ], admin_url('admin-post.php')),
                                        'lht_sl_delete_' . $r->id
                                    );
                                    ?>
                                    <a class="button button-small button-danger" href="<?php echo esc_url($delete_url); ?>"
                                        onclick="return confirm('Xoá short link này? Hành động không thể hoàn tác.');">
                                        Xoá
                                    </a>
                                </td>
                            </tr>
                        <?php endforeach; else: ?>
                        <tr>
                            <td colspan="7">Chưa có shortlink nào.</td>
                        </tr>
                    <?php endif; ?>
                </tbody>
            </table>

            <?php if ($pagination): ?>
                <div class="tablenav">
                    <div class="tablenav-pages">
                        <?php echo $pagination; ?>
                    </div>
                </div>
            <?php endif; ?>

            <hr>

            <!-- Cài đặt plugin -->
            <h2>Cài đặt plugin</h2>
            <form method="post" action="<?php echo esc_url(admin_url('options.php')); ?>">
                <?php settings_fields('lht_sl_settings'); ?>
                <table class="form-table">
                    <tr>
                        <th><label for="lht_sl_homepage_redirect">Redirect trang chủ đến URL</label></th>
                        <td>
                            <input name="<?php echo esc_attr(self::OPTION_HOMEPAGE_REDIRECT); ?>" id="lht_sl_homepage_redirect"
                                type="text" class="regular-text" value="<?php echo esc_attr($homepage_redirect); ?>"
                                placeholder="https://domain-khac.com">
                            <p class="description">
                                Để trống nếu không muốn redirect trang chủ.<br>
                                Nên trỏ sang domain khác để tránh loop.
                            </p>
                        </td>
                    </tr>
                    <tr>
                        <th>Khôi phục sạch dữ liệu khi xoá plugin</th>
                        <td>
                            <label>
                                <input type="checkbox" name="<?php echo esc_attr(self::OPTION_DELETE_ON_UNINSTALL); ?>"
                                    value="1" <?php checked($delete_on_uninstall, true); ?>>
                                Khi <strong>Delete plugin</strong>, hãy <strong>xóa toàn bộ dữ liệu</strong>
                                (bảng short link + các thiết lập).
                            </label>
                            <p class="description">
                                Cẩn thận: Không thể hoàn tác sau khi xoá plugin nếu tùy chọn này được bật.
                            </p>
                        </td>
                    </tr>
                </table>
                <p class="submit"><button class="button button-secondary">Lưu cài đặt</button></p>
            </form>
        </div>
        <?php
    }

    /** Lưu / cập nhật shortlink (tạo mới + sửa) */
    public function handle_save()
    {
        if (!current_user_can('manage_options'))
            wp_die('No permission');

        if (
            !isset($_POST['lht_sl_nonce']) ||
            !wp_verify_nonce($_POST['lht_sl_nonce'], 'lht_sl_save')
        ) {
            wp_die('Invalid nonce');
        }

        global $wpdb;
        $table = $wpdb->prefix . self::TABLE;

        $id = isset($_POST['id']) ? intval($_POST['id']) : 0;
        $long = trim($_POST['long_url'] ?? '');
        $code = trim($_POST['code'] ?? '');

        if (!$long)
            wp_die('Thiếu URL gốc');

        if (!preg_match('#^https?://#i', $long)) {
            $long = 'https://' . $long;
        }
        if (!filter_var($long, FILTER_VALIDATE_URL)) {
            wp_die('URL không hợp lệ');
        }

        if ($code !== '') {
            if (!preg_match('/^[a-zA-Z0-9\-]+$/', $code)) {
                wp_die('Code chỉ được dùng chữ, số, dấu "-"');
            }
        } else {
            $code = $this->generate_code();
        }

        // Check trùng slug post/page
        if (get_page_by_path($code, OBJECT, ['post', 'page'])) {
            wp_die('Code này trùng slug bài viết/trang, hãy chọn code khác.');
        }

        // Check trùng code trong bảng (ngoại trừ chính nó khi sửa)
        if ($id > 0) {
            $exists = $wpdb->get_var(
                $wpdb->prepare(
                    "SELECT COUNT(*) FROM $table WHERE code = %s AND id != %d",
                    $code,
                    $id
                )
            );
        } else {
            $exists = $wpdb->get_var(
                $wpdb->prepare(
                    "SELECT COUNT(*) FROM $table WHERE code = %s",
                    $code
                )
            );
        }

        if ($exists) {
            wp_die('Code này đã tồn tại, hãy chọn code khác.');
        }

        if ($id > 0) {
            // UPDATE
            $wpdb->update(
                $table,
                [
                    'code' => $code,
                    'long_url' => $long,
                ],
                ['id' => $id],
                ['%s', '%s'],
                ['%d']
            );
        } else {
            // INSERT
            $wpdb->insert(
                $table,
                [
                    'code' => $code,
                    'long_url' => $long,
                    'click_count' => 0,
                    'created_at' => current_time('mysql'),
                ],
                ['%s', '%s', '%d', '%s']
            );
        }

        wp_redirect(admin_url('admin.php?page=lht-short-links'));
        exit;
    }

    /** Xoá shortlink */
    public function handle_delete()
    {
        if (!current_user_can('manage_options'))
            wp_die('No permission');

        $id = isset($_GET['id']) ? intval($_GET['id']) : 0;
        if (!$id)
            wp_die('Thiếu ID');

        if (!wp_verify_nonce($_GET['_wpnonce'] ?? '', 'lht_sl_delete_' . $id)) {
            wp_die('Invalid nonce');
        }

        global $wpdb;
        $table = $wpdb->prefix . self::TABLE;

        $wpdb->delete($table, ['id' => $id], ['%d']);

        wp_redirect(admin_url('admin.php?page=lht-short-links'));
        exit;
    }

    /** Sinh code ngẫu nhiên */
    private function generate_code($len = 6)
    {
        $chars = 'abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789';
        $max = strlen($chars) - 1;
        $s = '';
        for ($i = 0; $i < $len; $i++) {
            $s .= $chars[random_int(0, $max)];
        }
        return $s;
    }

    /** Redirect shortlink + redirect trang chủ */
    public function handle_redirect()
    {
        // 1) Redirect trang chủ nếu có cấu hình (và user không login)
        if (is_front_page() && !is_admin() && !is_user_logged_in()) {
            $target = get_option(self::OPTION_HOMEPAGE_REDIRECT, '');
            if ($target && filter_var($target, FILTER_VALIDATE_URL)) {
                $h1 = parse_url(home_url(), PHP_URL_HOST);
                $h2 = parse_url($target, PHP_URL_HOST);
                if ($h1 !== $h2) { // tránh loop domain
                    wp_redirect($target, 301);
                    exit;
                }
            }
        }

        // 2) Xử lý shortlink dạng /code
        $code = get_query_var('lht_sl_code');
        if (!$code)
            return;

        global $wpdb;
        $table = $wpdb->prefix . self::TABLE;

        $row = $wpdb->get_row(
            $wpdb->prepare("SELECT * FROM $table WHERE code = %s", $code)
        );

        if (!$row) {
            // Nếu không có code, cho về trang chủ
            wp_redirect(home_url('/'), 302);
            exit;
        }

        // Tăng click
        $wpdb->query(
            $wpdb->prepare(
                "UPDATE $table SET click_count = click_count + 1 WHERE id = %d",
                $row->id
            )
        );

        wp_redirect($row->long_url, 302);
        exit;
    }

    /** Hàm uninstall – chạy khi Delete plugin */
    public static function uninstall()
    {
        // Chỉ dọn DB nếu admin đã bật checkbox
        $delete = get_option(self::OPTION_DELETE_ON_UNINSTALL, 0);
        if (!$delete) {
            return;
        }

        global $wpdb;
        $table = $wpdb->prefix . self::TABLE;

        // Xóa bảng
        $wpdb->query("DROP TABLE IF EXISTS $table");

        // Xóa options
        delete_option(self::OPTION_HOMEPAGE_REDIRECT);
        delete_option(self::OPTION_DELETE_ON_UNINSTALL);
    }
}

// Khởi tạo plugin
$lht_short_links_instance = new LHT_Short_Links();

// Đăng ký uninstall hook (static)
register_uninstall_hook(__FILE__, ['LHT_Short_Links', 'uninstall']);
